<?php
session_start();
include_once 'config/db.php';

if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: admin_login.php');
    exit;
}

$results_per_page = 10;

$stmt_total = $pdo->query("SELECT COUNT(id) FROM users");
$total_customers = $stmt_total->fetchColumn();

$total_pages = ceil($total_customers / $results_per_page);

$current_page = isset($_GET['page']) ? $_GET['page'] : 1;

$start_from = ($current_page - 1) * $results_per_page;

// Updated query with ORDER BY to sort customers by date in descending order
$stmt = $pdo->prepare("SELECT id, profile_link, customer_name, date, contact_no, registration_no, mileage FROM users ORDER BY created_at DESC LIMIT :start_from, :results_per_page");
$stmt->bindParam(':start_from', $start_from, PDO::PARAM_INT);
$stmt->bindParam(':results_per_page', $results_per_page, PDO::PARAM_INT);
$stmt->execute();
$customers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch updates in descending order of 'created_at'
$stmt_updates = $pdo->query("SELECT id, title, description, created_at FROM updates ORDER BY created_at DESC LIMIT 5");
$updates = $stmt_updates->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <?php include('partials/head.php'); ?>
</head>
<body>
    <div id="global-loader">
        <div class="whirly-loader"></div>
    </div>

    <div class="main-wrapper">
        <?php include('partials/header.php'); ?>
        <?php include('partials/sidebar.php'); ?>

        <div class="page-wrapper" style="max-height: 100vh; overflow-y: auto;">
            <div class="content">
                <div class="page-header mb-3">
                    <div class="page-title">
                        <h4>Recent Customer Data</h4>
                        <h6>New customers and recent updates</h6>
                    </div>
                </div>

                <div class="row mb-5" style="margin-bottom: 100px;">
                    <div class="col-xl-6">
                        <div class="card">
                            <div class="card-header justify-content-between">
                                <div class="card-title">
                                    Recently Created Customers
                                </div>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table text-nowrap">
                                        <thead class="table-primary">
                                            <tr>
                                                <th scope="col">Customer Name</th>
                                                <th scope="col">Date Added</th>
                                                <th scope="col">Contact</th>
                                                <th scope="col">Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($customers as $customer): ?>
                                                <tr>
                                                    <th scope="row"><?php echo htmlspecialchars($customer['customer_name']); ?></th>
                                                    <td><?php echo htmlspecialchars($customer['date']); ?></td>
                                                    <td><?php echo htmlspecialchars($customer['contact_no']); ?></td>
                                                    <td>
                                                        <div class="hstack gap-2 fs-15">
                                                            <a href="edit_customer.php?id=<?php echo $customer['id']; ?>" class="btn btn-icon btn-sm btn-soft-info rounded-pill"><i class="feather-edit"></i></a>
                                                             <a href="profile.php?link=<?php echo htmlspecialchars($customer['profile_link']); ?>" class="btn btn-icon btn-sm btn-soft-info rounded-pill"><i class="feather-user"></i></a>
                                                            <a href="delete_customer.php?id=<?php echo $customer['id']; ?>" class="btn btn-icon btn-sm btn-soft-danger rounded-pill"><i class="feather-trash"></i></a>
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>

                                <nav aria-label="Page navigation" class="m-2 pb-3">
                                    <ul class="pagination justify-content-center mb-3">
                                        <li class="page-item <?php echo ($current_page <= 1) ? 'disabled' : ''; ?>">
                                            <a class="page-link" href="<?php echo ($current_page > 1) ? '?page=' . ($current_page - 1) : 'javascript:void(0);'; ?>">Previous</a>
                                        </li>

                                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                            <li class="page-item <?php echo ($i == $current_page) ? 'active' : ''; ?>">
                                                <a class="page-link" href="?page=<?php echo $i; ?>"><?php echo $i; ?></a>
                                            </li>
                                        <?php endfor; ?>

                                        <li class="page-item <?php echo ($current_page >= $total_pages) ? 'disabled' : ''; ?>">
                                            <a class="page-link" href="<?php echo ($current_page < $total_pages) ? '?page=' . ($current_page + 1) : 'javascript:void(0);'; ?>">Next</a>
                                        </li>
                                    </ul>
                                </nav>
                            </div>
                        </div>
                    </div>

                    <div class="col-xl-6">
                        <div class="card">
                            <div class="card-header justify-content-between">
                                <div class="card-title">
                                    Recent Updates
                                </div>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table text-nowrap">
                                        <thead class="table-primary">
                                            <tr>
                                                <th scope="col">Title</th>
                                                <th scope="col">Description</th>
                                                <th scope="col">Date</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($updates as $update): ?>
                                                <tr>
                                                    <th scope="row"><?php echo htmlspecialchars($update['title']); ?></th>
                                                    <td><?php echo htmlspecialchars($update['description']); ?></td>
                                                    <td><?php echo htmlspecialchars($update['created_at']); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>

                </div>
            </div>
        </div>
    </div>

    <?php include('partials/footer.php'); ?>
</body>
</html>
